import os
import subprocess
import yaml
import sys
import time
import socket
import shutil
from pathlib import Path

# Constants
CONFIG_PATH = Path("settings.yaml")
TUNNEL_TIMEOUT = 10  # Seconds to wait for SSH tunnel
REQUIRED_TOOLS = ["ssh", "xfreerdp"]

DEFAULT_CONFIG = {
    "connection": {"name": "Anthro-Windows-VPS"},
    "ssh": {
        "host": "YOUR_VPS_IP",
        "port": 22,
        "user": "sshuser",
        "key": "~/.ssh/id_rsa",
    },
    "rdp": {
        "windows_host": "127.0.0.1",
        "local_port": 3390,
        "user": "Administrator",
        "password": "CHANGE_ME",
        "start_resolution": "1024x768",
        # "fullscreen_resolution": "1920x1080", # Optional
    },
    "performance": {
        "network_profile": "lan",
        "compression": False,
        "gfx": True,
    },
}

def check_dependencies():
    """Ensure required system tools are available."""
    missing = [tool for tool in REQUIRED_TOOLS if shutil.which(tool) is None]
    if missing:
        print(f"Error: Missing required tools: {', '.join(missing)}")
        sys.exit(1)

def load_or_create_config():
    """Load config with basic schema validation."""
    if not CONFIG_PATH.exists():
        try:
            with open(CONFIG_PATH, "w") as f:
                yaml.safe_dump(DEFAULT_CONFIG, f)
            print(f"{CONFIG_PATH} created. Please edit it and run again.")
            sys.exit(0)
        except IOError as e:
            print(f"Error writing config: {e}")
            sys.exit(1)

    try:
        with open(CONFIG_PATH) as f:
            cfg = yaml.safe_load(f)
    except yaml.YAMLError as e:
        print(f"Error parsing YAML: {e}")
        sys.exit(1)

    # Basic Validation
    required_sections = ["ssh", "rdp", "performance"]
    for section in required_sections:
        if section not in cfg:
            print(f"Config Error: Missing section '{section}'")
            sys.exit(1)
            
    return cfg

def wait_for_tunnel(local_port, process, timeout=TUNNEL_TIMEOUT):
    """
    Polls the local port to see if the SSH tunnel is active.
    Checks if the SSH process died prematurely.
    """
    start_time = time.time()
    print(f"Waiting for tunnel on port {local_port}...")

    while time.time() - start_time < timeout:
        # 1. Check if SSH process is still running
        if process.poll() is not None:
            print(f"SSH Error: Tunnel process died unexpectedly (Exit Code: {process.returncode}).")
            return False

        # 2. Try connecting to the forwarded port
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
            s.settimeout(0.5)
            try:
                if s.connect_ex(('127.0.0.1', local_port)) == 0:
                    print("Tunnel established.")
                    return True
            except socket.error:
                pass
        
        time.sleep(0.5)

    print("Error: Tunnel timed out. Port did not open.")
    return False

def start_ssh_tunnel(cfg):
    ssh = cfg["ssh"]
    rdp = cfg["rdp"]
    
    key_path = os.path.expanduser(ssh["key"])
    if not os.path.exists(key_path):
        print(f"SSH Error: Key file not found at {key_path}")
        sys.exit(1)

    cmd = [
        "ssh",
        "-N", # Do not execute a remote command
        "-L", f"{rdp['local_port']}:{rdp['windows_host']}:3389",
        "-p", str(ssh["port"]),
        "-i", key_path,
        "-o", "ExitOnForwardFailure=yes", # Exit if port binding fails
        "-o", "StrictHostKeyChecking=accept-new", # Convenience for VPS changes
        f"{ssh['user']}@{ssh['host']}",
    ]

    # Start SSH non-blocking
    return subprocess.Popen(cmd)

def start_rdp(cfg):
    rdp = cfg["rdp"]
    perf = cfg["performance"]

    # Build command list conditionally for cleanliness
    cmd = [
        "xfreerdp",
        f"/v:127.0.0.1:{rdp['local_port']}",
        f"/u:{rdp['user']}",
        f"/p:{rdp['password']}",
        f"/size:{rdp['start_resolution']}",
        "/dynamic-resolution",
        "/audio-mode:0",
        "/microphone",
        "/usb:auto",
        f"/network:{perf['network_profile']}",
        "/cert:ignore",
    ]

    if not perf.get("compression", False):
        cmd.append("/compression-level:0")
    
    if perf.get("gfx", True):
        cmd.append("/gfx") # AVC444 for Win Server 2022

    print("Starting FreeRDP...")
    try:
        subprocess.run(cmd, check=True)
    except subprocess.CalledProcessError as e:
        print(f"RDP exited with error code {e.returncode}")
    except KeyboardInterrupt:
        print("\nStopping RDP...")

def main():
    check_dependencies()
    cfg = load_or_create_config()
    
    print(f"Target: {cfg['connection'].get('name', 'Windows VPS')}")

    tunnel = start_ssh_tunnel(cfg)

    try:
        if wait_for_tunnel(cfg["rdp"]["local_port"], tunnel):
            start_rdp(cfg)
        else:
            print("Aborting RDP launch due to tunnel failure.")
    finally:
        if tunnel.poll() is None:
            print("Closing SSH tunnel...")
            tunnel.terminate()
            tunnel.wait()

if __name__ == "__main__":
    main()
